import tkinter as tk
from tkinter import filedialog, messagebox, ttk, scrolledtext
import os
import logging
from datetime import datetime
import re
import xml.etree.ElementTree as ET
import requests
from PIL import Image, ImageTk



#Definicion de colores para los distintas secciones
COLOR_FONDO = "#2c004b"   # Fondo general
COLOR_TEXTO = "#d27960"   # Texto
COLOR_BOTON = "#ffb200"   # Botones
 
class App(tk.Tk):

    def __init__(self):
        #diseño inicial
        super().__init__()
        self.configure(bg=COLOR_FONDO)
        self.title("NAPSE - Versión 7.8 - Envio de JSON por Servicio")
        self.geometry("700x700")

        self.archivo_path = tk.StringVar()
        self.tipo_var = tk.StringVar()
        self.registros_var = tk.StringVar(value="Maximo 2000 Registros")
        self.url_var = tk.StringVar(value="https//[IP o HOST]:API")

        self.fecha_actual = datetime.now().strftime('%Y%m%d')
        self.base_dir = os.path.dirname(os.path.abspath(__file__))
        self.carpeta_entrada = os.path.join(self.base_dir, "Entrada")
        self.carpeta_salida = os.path.join(self.base_dir, "Salida")
        self.carpeta_error = os.path.join(self.base_dir, "Error")
        self.carpeta_log = os.path.join(self.base_dir, "Log")

        os.makedirs(self.carpeta_entrada, exist_ok=True)
        os.makedirs(self.carpeta_salida, exist_ok=True)
        os.makedirs(self.carpeta_error, exist_ok=True)
        os.makedirs(self.carpeta_log, exist_ok=True)




        self.configurar_logger()
        self.crear_widgets()

         # Crear archivo de ejemplo lote.txt si no existe
        archivo_ejemplo = os.path.join(self.carpeta_entrada, "lote.txt")
        if not os.path.isfile(archivo_ejemplo):
            contenido = '''<customer type="bean"><code>75873381</code><store>0</store><party type="bean"><firstName>Pruebas3</firstName><lastName>Inicial Cliente3</lastName><genderType>F</genderType><typeCode>PRS</typeCode><birthCountryCode>ARG</birthCountryCode><birthDayNumber>22</birthDayNumber><birthMonthNumber>09</birthMonthNumber><birthYearNumber>1980</birthYearNumber><roleAssignments type="list"><partyRoleAssignment type="bean"><partyRole id="1"><name>Cliente</name></partyRole><effectiveDate format="dd-MM-yyyy">23-08-2017</effectiveDate><expirationDate format="dd-MM-yyyy">30-12-2200</expirationDate><contactMethods type="list"><partyContactMethod type="bean"><purposeType id="2"/><methodType id="1"/><effectiveDate>2017-08-23T16:32:37.947</effectiveDate><expirationDate>2200-12-30 00:00:00.000</expirationDate><address type="bean"> <city type="bean"><code>7346</code></city><country type="bean"> <code>PER</code></country><state type="bean"><code>7</code></state><firstLine>AVDA SAN MARTIN 486</firstLine><postalCode>3206</postalCode></address><emailAddress type="bean"><name>alejandro.silva@napse.global</name></emailAddress><telephone type="bean"><areaCode>11</areaCode><telephoneNumber>23878069</telephoneNumber><countryCode>54</countryCode></telephone></partyContactMethod></contactMethods><sequenceNumber>1</sequenceNumber></partyRoleAssignment></roleAssignments><identifications type="list"><partyIdentification type="bean"><identificationType id="1"/><identifierValue>75873380</identifierValue></partyIdentification></identifications></party>'''
            with open(archivo_ejemplo, "w", encoding="utf-8") as f:
                f.write(contenido)
            self.escribir_log("📄 Archivo de ejemplo 'lote.txt' creado en carpeta Entrada.")

        # Mostrar rutas de carpetas en el log al iniciar la aplicación
        self.escribir_log(f"📂 Carpeta Entrada: {self.carpeta_entrada}")
        self.escribir_log(f"📂 Carpeta Salida: {self.carpeta_salida}")
        self.escribir_log(f"📂 Carpeta Log: {self.carpeta_log}")
        self.escribir_log(f"📂 Carpeta Error: {self.carpeta_error}")


    #Configuracion log
    def configurar_logger(self):
        fecha_log = datetime.now().strftime("%Y-%m-%d")
        ruta_log_info = os.path.join(self.carpeta_log, f"log_{fecha_log}.txt")
        ruta_log_error = os.path.join(self.carpeta_error, f"Error_log_{self.fecha_actual}.txt")

        self.logger_info = logging.getLogger("info_logger")
        handler_info = logging.FileHandler(ruta_log_info, mode="a", encoding="utf-8")
        handler_info.setFormatter(logging.Formatter("%(asctime)s - %(levelname)s - %(message)s"))
        self.logger_info.setLevel(logging.INFO)
        self.logger_info.addHandler(handler_info)

        self.logger_error = logging.getLogger("error_logger")
        handler_error = logging.FileHandler(ruta_log_error, mode="a", encoding="utf-8")
        handler_error.setFormatter(logging.Formatter("%(asctime)s - %(levelname)s - %(message)s"))
        self.logger_error.setLevel(logging.ERROR)
        self.logger_error.addHandler(handler_error)

    def escribir_log(self, mensaje, nivel="info"):
        self.log_area.configure(state="normal")
        self.log_area.insert(tk.END, mensaje + "\n")
        self.log_area.configure(state="disabled")
        self.log_area.see(tk.END)
        if nivel == "info":
            self.logger_info.info(mensaje)
        elif nivel == "error":
            self.logger_error.error(mensaje)
            self.logger_info.error(mensaje)
        elif nivel == "warning":
            self.logger_info.warning(mensaje)

    def crear_widgets(self):
        titulo = tk.Label(self, text="Programa de Envío Automático por Servicio", 
                          bg=COLOR_FONDO, fg=COLOR_TEXTO, 
                          font=("Arial", 16, "bold"))
        titulo.pack(pady=(10, 20))
        frame = tk.Frame(self,bg=COLOR_FONDO)
        frame.pack(pady=10, padx=10, fill=tk.X)


        #Buscar archivo
        tk.Label(frame, text="Abrir lote.txt:", bg=COLOR_FONDO, fg=COLOR_TEXTO).grid(row=0, column=0, sticky="w")
        tk.Entry(frame, textvariable=self.archivo_path, width=60 ).grid(row=0, column=1)
        tk.Button(frame, text="Buscar", command=self.seleccionar_archivo , bg=COLOR_BOTON, fg=COLOR_FONDO).grid(row=0, column=2, padx=5)
        
        #Tipo servicio opcion
        tk.Label(frame, text="Tipo de servicio :",bg=COLOR_FONDO, fg=COLOR_TEXTO).grid(row=1, column=0, sticky="w")
        opciones = [
            "1 - Items", "2 - Clientes", "3 - Proveedor", "4 - Marca", "5 - Jerarquía (Level)",
            "7 - Unidad de Medida", "8 - Depósito", "9 - Atributo del Item", "10 - Código de Barra",
            "11 - Artículos de tipo Stock", "12 - Kits/enlazados", "13 - Artículos relacionados",
            "14 - Relación artículos con proveedores", "15 - Lista de Precios",
            "16 - Impuestos internos por artículo", "17 - Stock"
        ]
        ttk.Combobox(frame, textvariable=self.tipo_var, values=opciones, width=57).grid(row=1, column=1, columnspan=2, sticky="w")

        #Reistros por archivo
        #tk.Label(frame, text="Registros por archivo :",bg=COLOR_FONDO, fg=COLOR_TEXTO).grid(row=2, column=0, sticky="w")
        #tk.Entry(frame, textvariable=self.registros_var ).grid(row=2, column=1, sticky="w")
 

        tk.Label(frame, text="URL BMC :",bg=COLOR_FONDO, fg=COLOR_TEXTO).grid(row=3, column=0, sticky="w")
        tk.Entry(frame, textvariable=self.url_var, width=60).grid(row=3, column=1)

        botones = tk.Frame(self,bg=COLOR_FONDO)
        botones.pack(pady=10)
        tk.Button(botones, text="Generar Archivos XML",bg=COLOR_BOTON, fg=COLOR_FONDO ,command=self.generar_archivos).pack(side="left", padx=10)
        tk.Button(botones, text="Enviar Archivos", bg=COLOR_BOTON, fg=COLOR_FONDO, command=self.enviar_archivos).pack(side="left", padx=10)

        self.log_area = scrolledtext.ScrolledText(
            self,
            height=25,
            width=120,
            state="disabled",
            bg=COLOR_FONDO,        # Fondo oscuro
            fg=COLOR_TEXTO,        # Texto claro
            insertbackground=COLOR_TEXTO  # Cursor del texto
        )
        self.log_area.pack(padx=10, pady=(10, 0), fill=tk.BOTH, expand=True)
 
        # Footer con firma abajo a la derecha

        footer_frame = tk.Frame(self, bg=COLOR_FONDO)
        footer_frame.pack(fill=tk.X, pady=(5, 10), padx=10)

        #Firma registrada
        firma = tk.Label(footer_frame, text="Desarrollado por Alejandro Silva", 
                         bg=COLOR_FONDO, fg=COLOR_TEXTO, 
                         font=("Arial", 9, "italic"))
        firma.pack(side="right")

    def seleccionar_archivo(self):
        archivo = filedialog.askopenfilename(title="Seleccionar lote.txt", filetypes=[("Archivos de texto", "*.txt")])
        if archivo:
            self.archivo_path.set(archivo)
            self.escribir_log(f"📄 Archivo seleccionado: {archivo}")

    def generar_archivos(self):
        archivo = self.archivo_path.get()
        tipo = self.tipo_var.get().strip()
        try:
            registros = int(self.registros_var.get())
        except ValueError:
            registros = 1000
            

        if not os.path.isfile(archivo):
            messagebox.showerror("Error", "Archivo no válido.")
            self.escribir_log("❌ Archivo no válido.", "error")
            return
        #mapeo de opciones de busqueda
        mapeo = {
            "1": ("Item", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>item</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><items type='list'>", "</item></items></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "2": ("cliente", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>customer</service><request><![CDATA[ <bridgeCoreRequest> <operation>CreateOrUpdate</operation> <params> <customers type='list'>", "</customer> </customers> </params> </bridgeCoreRequest> ]]></request><store>100000003</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "3": ("Proveedor", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>supplier</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><suppliers type='list'>", "</supplier></suppliers></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "4": ("Marca", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>brand</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><brands type='list'>", "</brand></brands></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "5": ("Jerarquia(Level)", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>merchandiseHierarchy</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><merchHierarchyLevels type='list'>", "</hierarchyLevel></merchHierarchyLevels></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "7": ("Unidad de Medida", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>unitOfMeasure</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><uoms type='list'>", "</uom></uoms></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "8": ("Deposito", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>location</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><locations type='list'>", "</location></locations></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "9": ("Atributo de Item", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>itemAttribute</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><itemAttributes type='list'>", "</itemAttribute></itemAttributes></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "10": ("Codigo de Barra", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>alias</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><aliases type='list'>", "</alias></aliases></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "11": ("Articulo de tipo Stock", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>stockItems</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><stockItems type='list'>", "</stockItem></stockItems></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "12": ("Kits Enlazados", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>linkedItem</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><linkedItem type='list'>", "</item></linkedItem></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "13": ("Articulos relacionados", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>itemRelated</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><itemsRelated type='list'>", "</itemRelated></itemsRelated></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "14": ("Relacion articulos con Proveedores", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>itemSupplierItem</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><itemSupplierItems type='list'>", "</itemSupplierItem></itemSupplierItems></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "15": ("Lista de Precios", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>itemSellingPrices</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><itemSellingPrices type='list'>", "</itemSellingPrices></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "16": ("Importacion impuestos internos por articulo", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>itemInternalTax</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><itemInternalTaxes type='list'>", "</itemInternalTax></itemInternalTaxes></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>"),
            "17": ("Stock", "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:ser=\"http://services.business.soap.bridge.synthesis.com/\"><soapenv:Header/><soapenv:Body><ser:execute><service>stock</service><request><![CDATA[<bridgeCoreRequest><operation>createOrUpdate</operation><params><stocks type='list'>", "</stock></stocks></params><replicationStore>0</replicationStore></bridgeCoreRequest>]]></request><store>0</store></ser:execute></soapenv:Body></soapenv:Envelope>")
        
        }

        codigo = tipo.split(" - ")[0].strip()
        if codigo in mapeo:
            entidad, texto1, texto2 = mapeo[codigo]
            self.dividir_archivo(archivo, registros, texto1, texto2, entidad)
            self.escribir_log(f"✅ Archivos generados para '{entidad}'.")
        else:
            self.escribir_log("❌ Tipo de servicio no implementado aún en la GUI.", "warning")

    def dividir_archivo(self, archivo_entrada, registros_por_archivo, texto1, texto2, entidad):
        try:
            contador_archivo = 1
            registros = []
            with open(archivo_entrada, 'r', encoding='utf-8') as archivo:
                for linea in archivo:
                    registros.append(linea)
                    if len(registros) == registros_por_archivo:
                        self.escribir_archivo_xml(registros, contador_archivo, texto1, texto2, entidad)
                        registros = []
                        contador_archivo += 1
            if registros:
                self.escribir_archivo_xml(registros, contador_archivo, texto1, texto2, entidad)
        except Exception as e:
            self.escribir_log(f"❌ Error dividiendo archivo: {e}", "error")

    def escribir_archivo_xml(self, registros, contador, texto1, texto2, entidad):
        nombre = f"{entidad}_lote_{self.fecha_actual}_pte_{contador}.xml"
        ruta = os.path.join(self.carpeta_salida, nombre)
        try:
            with open(ruta, 'w', encoding='utf-8') as salida:
                salida.write(texto1 + '\n')
                salida.writelines(registros)
                salida.write('\n' + texto2)
            self.escribir_log(f"✅ Archivo creado: {nombre}")
            self.escribir_log(f"   ➡ Cantidad de registros en archivo: {len(registros)}")
        except Exception as e:
            self.escribir_log(f"❌ Error escribiendo {nombre}: {e}", "error")

    def enviar_archivos(self):
        url = self.url_var.get().strip()
        if not url.startswith("http") or ":" not in url:
            self.escribir_log("❌ URL inválida. Debe tener el formato http(s)://host:puerto", "error")
            return

        final_url = url + "/bridge/services/bridgeCoreSOAP"
        self.escribir_log(f"🔗 URL Final: {final_url}")

        archivos = [f for f in os.listdir(self.carpeta_salida) if f.endswith(".xml") and "lote" in f]
        if not archivos:
            self.escribir_log("⚠️ No hay archivos para enviar.", "warning")
            return

        for archivo in archivos:
            path = os.path.join(self.carpeta_salida, archivo)
            try:
                with open(path, "r", encoding="utf-8") as f:
                    contenido = f.read()
                headers = {'Content-Type': 'text/xml; charset=utf-8', 'SOAPAction': ''}
                response = requests.post(final_url, data=contenido, headers=headers, verify=False)
                if response.status_code == 200:
                    self.escribir_log(f"📤 {archivo} enviado correctamente.")
                    os.remove(path)
                else:
                    self.escribir_log(f"❌ Error {response.status_code} al enviar {archivo}.", "error")
            except Exception as e:
                self.escribir_log(f"❌ Error enviando {archivo}: {e}", "error")

if __name__ == "__main__":
    app = App()
    app.mainloop()
    